classdef NOVUDP < handle

   properties
   
      type = "UDP";
      version = "1.0.0";
      IPADDR;
      PORT = 5024;
      Handle;
      IsOctave = 0;
      isConnected = 0;

   end


   methods
   
      function obj = NOVUDP(IPADDR)
      
         %disp(sprintf('IPADDR = %s;', IPADDR));
         obj.IPADDR = IPADDR;
         init(obj);
         
      end;
      
      function s = getip(obj)
         s = obj.IPADDR;
      end;

      function init(obj)
      
         if exist('OCTAVE_VERSION', 'builtin') % Octave
            pkg load instrument-control
            obj.IsOctave = 1;
            
            %set(obj.Handle, 'InputBufferSize', 1024)
            
         end;
         obj.Handle = udpport;
         
         %obj.Handle.ByteOrder = "big-endian" % works only for Matlab
         obj.Handle.Timeout = 2;
         
         obj.isConnected = 1;

      end;
      
      function close(obj)
         obj.Handle = 0;
         clear obj.Handle;
         obj.isConnected = 0;
      end;
      
      function crc = crc16(obj, crc, data)
         for ii=1:length(data),
            crc = bitand( bitor( bitshift(crc, 1), bitshift(crc, -15) ), 2^16-1);
            crc = bitxor( crc, data(ii) );
         end;
      end;
      
      function data = addcrc16(obj, data)
         crc = 2^16-1;
         for ii=1:length(data);
            crc_bin = dec2bin(crc, 16);
            crc_shifted = bin2dec(crc_bin([2:16 1]));
            crc = bitxor(data(ii), crc_shifted);
         end;
         data = [data crc];
      end;
      
      function ok = write(obj, addr, data)
      
         % write packet: |  0 'W' | addr | data | CRC |
         % ack packet:   | 0xaaaa | XXXX | 
      
         flush(obj.Handle);
      
         seq1 = [87 addr data 0 0];
         crc  = crc16(obj, 2^16-1, seq1);
         %disp(sprintf("\tcrc: 0x%04x", crc));
         write(obj.Handle, [seq1 crc], "uint16", obj.IPADDR, obj.PORT);
         
         res = read(obj.Handle, 2, "uint16");
         
         %for ii=1:2
         %   disp(sprintf("\tres%d: 0x%04x", ii, res(ii)));
         %end;
         
         ok = 0;
         switch res(1)
            case hex2dec('aaaa'), ok = 1;
            case hex2dec('cccc'), disp('CRC ERROR during UDP write');
            case hex2dec('eeee'), disp('COMMAND ERROR during UDP write');
            otherwise disp('UNKNOWN ERROR during UDP write');
         end;
      end;
      
      
      function [res ok] = read(obj, addr)
      
         % read packet: |  0 'R' |  addr  |   0 | CRC |
         % resp packet: | 0xaaaa | CRCNEW | data | FIFOCRC |      xxx data und CRCNEW tauschen??


         %tic;
         flush(obj.Handle);
         %toc, tic;

         if length(addr)==1,
            seq1 = [uint16('R') addr 0 0 0];
         else
            %seq1 = [82 reshape(addr, [1 length(addr)])]; % no function yet!!!
         end;
         crc  = crc16(obj, 2^16-1, seq1);
         %toc, tic;
         
         %seq1
         
         % send read request
         write(obj.Handle, [seq1 crc], "uint16", obj.IPADDR, obj.PORT);
         %toc, tic;
         
         starttic = tic;
         BytesAvailable = obj.Handle.NumBytesAvailable;
         while (BytesAvailable<8)&&(toc(starttic)<1),
            BytesAvailable = obj.Handle.NumBytesAvailable;
         end;
         
         %BytesAvailable
         
         ok = 0;
         % read response
         if BytesAvailable>=8,
         
            recv = read(obj.Handle, 4, "uint16");
            
            res = [];
            switch recv(1)
               case hex2dec('aaaa'), ok = 1; res = recv(3);
               case hex2dec('cccc'), disp('CRC ERROR during UDP read');
               case hex2dec('eeee'), disp('COMMAND ERROR during UDP read');
               otherwise disp('UNKNOWN ERROR during UDP read');
            end;
            
            % analyze crc in response
            crcnew = crc16(obj, crc, recv(3));
            if crcnew~=recv(2), % wrong crc in response
               disp('WRONG CRC in UDP response');
               disp(sprintf("\tcrc:    0x%04x", crc));
               disp(sprintf("\tcrcnew: 0x%04x", crcnew));
               disp(sprintf("\tcrcrec: 0x%04x", recv(2)));
               disp(sprintf("\tdata:   0x%04x", recv(3)));
               ok = 0;
               res = [];
             end;
            
         else
            res = [];
            disp('Timeout during read in NOVUDP.m');
         end;
         

      end;
      
      
       % only for PM1000: Reads data from SDRAM
      function [res, ok, status] = readhspm(obj, startaddr, numaddr)
      
         if obj.IsOctave,
            bufsizeaddr = 2^13;
         else
            bufsizeaddr = 2^15;
         end;
            
         addrtransferred = 0;
         %res = [];
         res = zeros(4, numaddr);
         while (addrtransferred<numaddr),
            numaddrhere   = min(bufsizeaddr, numaddr-addrtransferred);
            startaddrhere = startaddr+addrtransferred;
            [reshere, ok, status] = readhspm_raw(obj, startaddrhere, numaddrhere);
            %size(reshere)
            if ok,
               %res = [res reshere];
               res(:,(addrtransferred+1):(addrtransferred+numaddrhere)) = reshere(:,1:numaddrhere);
               addrtransferred = addrtransferred + numaddrhere;
            else
               break;
            end;
         end;
      end;
      
      function [res, ok, status] = readhspm_raw(obj, startaddr, numaddr)
         
         numaddr = min(256*512, numaddr);
         numblocks = ceil(numaddr/512);

         %seq1 = [uint16('S') + (numblocks-1)*2^8 bitshift(startaddr, -16) mod(startaddr, 2^16)];
         seq1 = [uint16('S') bitshift(startaddr, -16) mod(startaddr, 2^16) (numblocks-1) 0];
         crc  = crc16(obj, 2^16-1, seq1);
         %toc, tic;
         
         ok = 0;
         tries = 0;
         while ok<1 && tries<5
         
            flush(obj.Handle);

            % send read request
            write(obj.Handle, [seq1 crc], "uint16", obj.IPADDR, obj.PORT);
            
         
            starttic = tic;
            BytesAvailable = obj.Handle.NumBytesAvailable;
            while (BytesAvailable<8*numaddr)&&(toc(starttic)<0.2), % expected time is <0.025 seconds
               BytesAvailable = obj.Handle.NumBytesAvailable;
            end;
            
            if BytesAvailable>=8*numaddr,
               ok = 1;
               %toc(starttic)
            else
               tries = tries + 1
            end;
            
         end;
         
         % read response
         if BytesAvailable>=8*numaddr,
            recv = read(obj.Handle, 4*numaddr, "uint16");
            res = reshape(recv, [4 numaddr]);
            ok = 1;
         else
            res = [];
            ok = 0;
            disp('Timeout during readhspm_raw in NOVUDP.m');
            numaddr
            BytesAvailable
         end;
         
         status = 1;
      end;
      
      
      function [res, ok, status] = readburst_raw(obj, wr_reg, startvalue, rd_reg)
         
         flush(obj.Handle);
         
         seq1 = [uint16('B') wr_reg startvalue 0 rd_reg];
         crc  = crc16(obj, 2^16-1, seq1);

         % send burst read request
         write(obj.Handle, [seq1 crc], "uint16", obj.IPADDR, obj.PORT);
         
         starttic = tic;
         BytesAvailable = obj.Handle.NumBytesAvailable;
         while (BytesAvailable<1024)&&(toc(starttic)<1),
            BytesAvailable = obj.Handle.NumBytesAvailable;
         end;
         
         % read response
         if BytesAvailable>=1024,
            res = read(obj.Handle, 512, "uint16");
            ok = 1;
         else
            res = 0;
            ok = 0;
            disp('Timeout during readburst in NOVUDP.m');
         end;
         
         status = 1;
      end;
      
      function [res, ok, status] = readburst(obj, wr_reg, startvalue, stopvalue, rd_reg)
         
         ok = 1;
         
         numaddresses = stopvalue - startvalue + 1;
         addrreceived = 0;
         
         while addrreceived<numaddresses,
            [reslocal, ok, status] = readburst_raw(obj, wr_reg, startvalue+addrreceived, rd_reg);
            numaddrhere = min(512, numaddresses-addrreceived);
            if ok>0,
               res((addrreceived+1):(addrreceived+numaddrhere))=reslocal(1:numaddrhere);
               addrreceived = addrreceived + numaddrhere;
            end;
         end;
         
         status = 1;
         
      end;
      
      
      
      
      % only for PM1000: Reads one block of the Poincaré sphere frame buffer
      function [res ok] = readfr(obj, BlockNr)
      
         TCPHandle = obj.Handle;
      
         NumBytes = 2^12; % one complete frame block

         debugmode=0;

         ok=0;

         tic
         
         flush(obj.Handle);

         %seq1 = [uint16('T') BlockNr 0];
         seq1 = [uint16('T')  0 BlockNr*2^11 0 0];
         crc  = crc16(obj, 2^16-1, seq1);

         % send read request
         write(obj.Handle, [seq1 crc], "uint16", obj.IPADDR, obj.PORT);
         %toc, tic;
         
         %pause(.1);
         
         
         %BytesAvailable = obj.Handle.NumBytesAvailable       
         %res = read(obj.Handle, BytesAvailable/2, "uint16");
         
         res = read(obj.Handle, 2^11, "uint16");
         
         if 0
         
         

            % request data stream
            write(obj, 512 + 105, BlockNr);
            pause(.001);
            write(obj, 512 + 104, 27273);
            pause(.001);


            BytesAvailable=0;
            StopCounter=0;
            while BytesAvailable<NumBytes & StopCounter<2000,
               BytesAvailable = TCPHandle.NumBytesAvailable;
               StopCounter = StopCounter+1;
               pause(.001);
            end;

            if debugmode
               toc % 10..200 ms with PM1000
            end;


            res=0;
            if BytesAvailable>=NumBytes,
               res = fread(TCPHandle, BytesAvailable);
               ok=1;
            else
               BytesAvailable
               StopCounter
            end;

            if debugmode
               BytesAvailable
            end
            
            
         end;
      end;
      
      
     
      
      
      
      function [res, ok, status] = readhspm_old(obj, startaddr, numaddr)
      
         status = 0;
         
         buffersizebytes=2^14;


         %cycles=max(1, min(32, numaddr*8/buffersizebytes))
         %cycles=max(1, min(32, numaddr*8/buffersizebytes)) % debug
         cycles=max(1, min(32, ceil(numaddr*8/buffersizebytes))); % debug
         cycles=2^ceil(log2(cycles));
         
         numaddr=ceil(numaddr/cycles)*cycles;


         buffersizeaddr=buffersizebytes/8*cycles;

         packets_transferred=0;

         res=[];

         tic;
            
         while packets_transferred<numaddr,
            
             packetsinthissequence=min(buffersizeaddr, numaddr-packets_transferred);
            
             startaddrseq = startaddr + packets_transferred;

             [res_seq, ok, timeoutreached] = PM1000_requestpackets(obj, startaddrseq, packetsinthissequence, cycles);
             
             if timeoutreached,
               break;
             end;
            
             if ok,
                 packets_transferred=packets_transferred+packetsinthissequence;
                 res=[res res_seq];
             else
                 disp('HS data transmission failed. Trying again...')
             end;
             
             
                 

         end;

         %toc
            
         %pause(.1); BytesAvailable = TCPHandle.NumBytesAvailable   

      end
      
      
      
      % only for PM1000: Requests one or more packets of data from SDRAM
      function [res, ok, timeoutreached] = PM1000_requestpackets(obj, startaddrseq, packetsinthissequence, cycles)

         TCPHandle = obj.Handle;
         

         % reqisters for HS-configuration
         addr_data = [ 512+105 mod(startaddrseq, 2^16);
                       512+106 bitshift(startaddrseq, -16) + log2(cycles)*2^12;
                       %512+107 packetsinthissequence;
                       512+107 round(packetsinthissequence/cycles);
                       512+104 39294]; % trigger code for HS-transfer over LAN
         for ii=1:4,
              seq(:,ii) = [87 bitshift(addr_data(ii,1), -8) mod(addr_data(ii,1), 2^8) bitshift(addr_data(ii,2), -8) mod(addr_data(ii,2), 2^8) ];
         end;
         
         %writeepstcp(512+105,0); % set address: lower 16 bits
         
         % send the request
         write(TCPHandle, char(reshape(seq, 1, 20))); % writing as one packet improves timing compared to single write packets
         
         timeouttic=tic;
         
         res=[];
         for ii=1:cycles,
            BytesAvailable=0;
            timeoutreached=0;
            while BytesAvailable<packetsinthissequence*8/cycles && timeoutreached==0,
                 pause(.001);
                 BytesAvailable = TCPHandle.NumBytesAvailable;
                 if toc(timeouttic)>2
                     timeoutreached=1;
                     disp('Timeout reached before data received!');
                     break;
                 end;
            end;
            
            if timeoutreached,
               break;
            end;
            %toc(timeouttic) % 17 ms for 512 addr
            %toc(timeouttic) % 622 ms for 2^16 addr
            
            if BytesAvailable>=packetsinthissequence*8/cycles, % all bytes in buffer
                 
               if exist('OCTAVE_VERSION', 'builtin') % Octave
                  B = read(TCPHandle, packetsinthissequence*8/cycles); % 2 Bytes per Sample
                  for ii=1:packetsinthissequence*4/cycles, % reverse byte order as long as Octave does not support big-endian
                     A(ii) = uint16(B((ii-1)*2+1))*256 + uint16(B((ii-1)*2+2));
                  end;
               else
                  A = read(TCPHandle, packetsinthissequence*4/cycles, 'uint16'); % 2 Bytes per Sample
               end;
                 res=[res reshape(A, 4,packetsinthissequence/cycles)];
                 ok=1;
             else
                 ok=0;
            end;
         end;
      end





         
         

   end;

end