classdef EPS1000

   properties
   
      model = "EPS1000"
      version = "1.1.2"
      
   end
   
   methods(Static)
   
      %%%%%%%%%%%%%%%%%%%%%%%
      % Basic communication %
      %%%%%%%%%%%%%%%%%%%%%%%

      function isConnected = init(ARG)
         % EPS1000.init('192.168.1.100');    opens LAN device
         % EPS1000.init(1);                  opens last USB device
         % EPS1000.init(0);                  shows list of USB devices and lets user chose one
         % EPS1000.init;                     same as eps=EPS1000(1);
         % PM1000.init;EPS1000('PMS');       connects to eps card inside a PMS1000;

         if (nargin<1)
            ARG=1; % connect to last device
         end;
         
         global EPSHandle
         if length(ARG)==3&&sum(ARG=='PMS')==3  % via PMS1000 device
            global PMHandle
            EPSHandle = PMHandle
         elseif length(ARG)>6&&sum(class(ARG)=='char')==4  % LAN DEVICE
            EPSHandle = NOVTCP(ARG);
         else % USB DEVICE
            EPSHandle = NOVUSB('LastDevEPS.mat', ARG);
            %EPSHandle.init('LastDevEPS.mat', ARG);
         end;
         
         isConnected = EPSHandle.isConnected;
         
      end;
      
      function close
         global EPSHandle
         EPSHandle.close
      end;
      
      function [res ok] = read(addr)
         
         global EPSHandle
         
         [res ok] = EPSHandle.read(addr);
         
         if ok==0, % su success in reading. Try to connect again
            EPS1000.init;
            [res ok] = EPSHandle.read(addr);
         end;
         
      end;
      
      function [res] = setbitmode(mask, mode)
         
         global EPSHandle
         
         [res] = EPSHandle.setbitmode(mask, mode);
         
      end;
	  
	  function [dout1 ok] = readburst(rdaddr, addrstart, addrstop, wraddr)
	  
         global EPSHandle
         [dout1 ok] = EPSHandle.readburst(rdaddr, addrstart, addrstop, wraddr);
         if ok==0, % su success in reading. Try to connect again
            EPS1000.init;
            [dout1 ok] = EPSHandle.readburst(rdaddr, addrstart, addrstop, wraddr);
         end;
		end;

      function ok = write(addr, data)
      
         global EPSHandle
         ok = EPSHandle.write(addr, data);
         if ok==0, % su success in writing. Try to connect again
            EPS1000.init;
            [res ok] = EPSHandle.write(addr, data);
         end;
         
         % pause to force gap between write commands
         pausetime = 0.0010; % pause 1 ms at FT232R
         mstic=tic;
         mspause=0;
         while mspause<pausetime,
            mspause=toc(mstic);
         end;
         
         
      end;
      
    
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%
      % General instrument data %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function r = getfirmware % as string
         r = sprintf('%4X', EPS1000.read(84));
      end
      
      
      function r = getserialnumber
         r = EPS1000.read(91);
      end;
      
      function r = getmoduletype % as string
         str=[];
         for ii=0:15,
            dummy = EPS1000.read(96+ii);
            str=[ str char(bitshift(dummy, -8)) ];
            str=[ str mod(dummy, 2^8) ];
         end;
         r = str;
      end
      
      function r = getmacaddress % as string
         str = [];
         for ii=0:5
            str = [str sprintf('%02x ',EPS1000.read(300+ii))];
         end;
         r = str;
      end;
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % Device specific functions %
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%

      function r = gettemperature
         r = EPS1000.read(181)/16;
      end;
      
      function r = getnrofwlenbands
         r = EPS1000.read(190);
      end;
      
      
      
      function r = getoptfrequency(freq) % in THz
         band = EPS1000.getband;
         minfreq = EPS1000.read(193 + 5*band)/2^7;
         maxfreq = EPS1000.read(192 + 5*band)/2^7;
         steps = EPS1000.read(194 + 5*band);
         stepsize = (maxfreq-minfreq)/steps;
         index = EPS1000.read(25)-1;
         freq = minfreq+stepsize*index;
         freq = round(10*freq)/10;
         r=freq;
      end;
      
      function setoptfrequency(freq) % in THz
         band = EPS1000.getband;
         minfreq = EPS1000.read(193 + 5*band)/2^7;
         maxfreq = EPS1000.read(192 + 5*band)/2^7;
         steps = EPS1000.read(194 + 5*band);
         stepsize = (maxfreq-minfreq)/steps;
         index = round((freq-minfreq)/stepsize)+1;
         if (index>=0) & (index<=steps)
            EPS1000.write(25, index);
         else
            disp('Freq. exceeds range!');
         end;
         
      end;
      
      function r = getband(band)
         r = EPS1000.read(26);
      end;
      
      function setband(band)
         EPS1000.write(26, band);
      end;
      
      function setposition(wp, position) % as integer 0..2^16-1
         %0: HWP
         %1..6: QWP0..QWP5
         EPS1000.write(40+wp, round(position));
      end;
      
      function r=getposition(wp) % as integer 0..2^16-1
         %0: HWP
         %1..6: QWP0..QWP5
         r = EPS1000.read(40+wp);
      end;
      
      function speed = getmaxhwpspeed
         speed = (bitshift(EPS1000.read(249), 16) + EPS1000.read(248)) * 10;
      end;
      
      function setspeed(wp, speed) % nominal speed in rad/s
      
         MAXHWP = EPS1000.getmaxhwpspeed;
         MAXQWP = max(2000000, MAXHWP/5);
         %0: HWP
         %1..6: QWP0..QWP5
         EPS1000.write(10+2*wp, 0); % clear MSB register first to avoid overflow
         
         if wp==0, % HWP
            speed = min(MAXHWP, speed);
            speed_fpga = round(speed/10);
         else
            speed = min(MAXQWP, speed);
            speed_fpga = round(speed*100);
         end;
         %speed_fpga
         EPS1000.write(9+2*wp, mod(speed_fpga, 2^16)); % LSB
         EPS1000.write(10+2*wp, bitshift(speed_fpga, -16)); % MSB

      end;
      
      function r=getspeed(wp) % nominal speed in rad/s
         %0: HWP
         %1..6: QWP0..QWP5
         r = EPS1000.read(10+2*wp)*2^16 + EPS1000.read(9+2*wp);
         if wp==0, % HWP
            r = r*10;
         else
            r = r/100;
         end;
      end;
      
      function setrotation(wp, backw_en)
         %0: HWP
         %1..6: QWP0..QWP5
         %0, 2: STOP
         %1, FORWARD
         %3, BACKWARD
         EPS1000.write(wp, backw_en);
      end;

      function setrotation_all(backw_en)
         %backw_en is a vector of 7 values for 7 waveplates
         val=0;
         for ii=1:length(backw_en)
            val = val + backw_en(ii)*2^(2*(ii-1));
         end;
         EPS1000.write(7, val);
      end;
      
      function setdistr_int(speed, distrcode)
         target_fpga = round(speed/10);
         EPS1000.write(24, 0); % clear MSB register to avoid overflow
         EPS1000.write(23, mod(target_fpga, 2^16));
         EPS1000.write(24, 1*2^15 + distrcode*2^13 + bitshift(target_fpga, -16));
         % test:
         % mod(EPS1000.read(24), 2^10)*2^16 + EPS1000.read(23)
      end;
      
      function setdistrpeaked(speed) % in rad/s, requires firmware >=1.2.0.7
         distrcode = 0;
         EPS1000.setdistr_int(speed, distrcode); 
      end;
      
      function setdistrpeakedplus(speed) % in rad/s, requires firmware >=1.2.3.0
         if 1
            distrcode = 3;
            EPS1000.setdistr_int(speed, distrcode);
         else
            % this shows how you set peakedplus manually with earlier firmware versions:
            distrcode = 0;
            next_cal_Mrads = min(5, max(2, ceil(speed/10000000)))*1e7
            
            EPS1000.setdistr_int(speed, distrcode);
            quot = speed/next_cal_Mrads;
            quot = quot + (1-quot)*0.08; % factor taken from experiments
            fact = asin(quot)/(pi/2)
            EPS1000.write(33, round(fact*2^15));
            EPS1000.write(34, round(fact*2^15));
            EPS1000.setspeed(0, next_cal_Mrads);
         end;
      end;
      
      function setdistrrayleigh(speed) % in rad/s, requires firmware >=1.2.0.7
         distrcode = 2;
         EPS1000.setdistr_int(speed, distrcode);
      end;
      
      function setdistrdepol(speed) % in rad/s, requires firmware >=1.2.2.6
         distrcode = 1;
         EPS1000.setdistr_int(speed, distrcode);
      end;
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%
      % Synchronous/Triggered %
      %%%%%%%%%%%%%%%%%%%%%%%%%
      
      function r=getexecutiontime_40ns % in steps of 40 ns
      
         r= EPS1000.read(222) + EPS1000.read(223) * 2^16;

      end
      
      function setexecutiontime_40ns(Execution_Time_Int) % in steps of 40 ns
      
         %Execution_Time_Int=25000; % 25000 * 40 ns = 1 ms
         %Execution_Time_Int=2500; % 2500 * 40 ns = 100 us
         %Execution_Time_Int=500; % 500 * 40 ns = 20 us
         EPS1000.write(222, mod(Execution_Time_Int, 2^16)); % lower 16 Bits
         EPS1000.write(223, bitshift(Execution_Time_Int, -16)); % upper 16 Bits

      end
      
      function setexecutiontime_us(Execution_Time_us) % in us
         EPS1000.setexecutiontime_40ns(25*Execution_Time_us);
      end

      
      
      %%%%%%%%%%%%%%%%%%%%%
      % Special functions %
      %%%%%%%%%%%%%%%%%%%%%
      
      function setfanhysteresis(hyst)
         EPS1000.write(180, round(hyst));
      end;
      
      
      
      % writes voltage table to EPS
      function WriteTable(V, dwell_time_ns, nvolt)

         % determine the number of table entries (rows)
         npos=size(V,1);
         
         if length(dwell_time_ns)==1,
            dwell_time_ns=ones(1, npos)*dwell_time_ns;
         end;
            
            
         if nvolt==0, % write all voltages of mvolt
            xvolt = 1:size(V,2);
         else
            xvolt = nvolt; % write only specified row of mvolt
         end;

         for ii=1:npos
            EPS1000.write(219, ii-1); % select row
            % dwell time in 40ns-ticks:
            dwell_time_ticks=max(4, round(dwell_time_ns(ii)/40)-1);
            EPS1000.write(250, mod(dwell_time_ticks, 2^16)); % lower 16 Bits
            EPS1000.write(251, bitshift(dwell_time_ticks, -16)); % upper 16 Bits
            trigger = 1;
            % table data:
            for iii=xvolt
               EPS1000.write(252+iii-1, V(ii,iii)); % store current electrode voltages in registers
               trigger = bitor(trigger, 2^(iii-1));
            end;
            % Table write trigger
            EPS1000.write(221, trigger);
         end;
         
      end;
      
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      %      Table Control
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function table_manualtrigger
         EPS1000.write(227, 1);
      end;
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      %      SOP Tracking
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function setcontrolenable(en)
         if en==1,
            EPS1000.write(340, 3); % this also sets tracking mode
         end;
         if en==0,
            dummy = EPS1000.read(340);
            EPS1000.write(340, dummy - mod(dummy,2)); % disable control but leave EPS in tracking mode
         end;
      end;
      
      function settrackingmode(en) % this overwrites the control enable bit
         en = min(en, 1);
         EPS1000.write(340, en*2);
      end;

      
      function showcontrolparameters()
         gain       = EPS1000.read(341);
         maximize   = EPS1000.read(342);
         ATE        = EPS1000.read(343);
         delay      = EPS1000.read(344);
         ditherampl = EPS1000.read(345);
         pullback   = EPS1000.read(346);
         mode = EPS1000.getcontrolmode();
         disp(sprintf('Detector Mode:        %d', mode.detector));
         disp(sprintf('Gradient Type:        %d', mode.gradtype));
         disp(sprintf('Reference Path:       %d', mode.reference));
         disp(sprintf('Sensitive Mode:       %d', mode.sensitive));
         disp(sprintf('APD Control:          %d', mode.apdcontrol));
         disp(sprintf('Control Gain:         %d', gain));
         disp(sprintf('Maximization:         %d', maximize));
         disp(sprintf('Averaging Time Exp.:  %d', ATE));
         disp(sprintf('Error Signal Delay:   %d', delay));
         disp(sprintf('Dither Amplitude:     %d', ditherampl));
         disp(sprintf('Pull Back Strength:   %d', pullback));
       end;
         
         
      
      function setcontrolgain(gain)
         EPS1000.write(341, gain);
      end;
      
      function setmaximize(maximize)
         maximize = min(maximize, 1);
         EPS1000.write(342, maximize);
      end;
      
      function setATE(ATE)
         EPS1000.write(343, ATE);
      end
      
      function setdelay(delay)
         EPS1000.write(344, delay);
      end
      
      function r=getdelay
         r = EPS1000.read(344);
      end
      
      function setrelmodstep(step)
         EPS1000.write(345, step);
      end
      
      function setrelpullstep(step)
         EPS1000.write(346, step);
      end
      
      function setcontrolmode(mode)
         EPS1000.write(347, mode.apdcontrol*2^6 + mode.sensitive*2^4 + mode.reference*2^3 + mode.gradtype*2^2 + mode.detector);
      end;
      
      function mode = getcontrolmode()
         dummy=EPS1000.read(347);
         mode.detector   = mod(dummy, 2^2);
         mode.gradtype   = mod(bitshift(dummy, -2), 2^1);
         mode.reference  = mod(bitshift(dummy, -3), 2^1);
         mode.sensitive  = mod(bitshift(dummy, -4), 2^2);
         mode.apdcontrol = mod(bitshift(dummy, -6), 2^1);
      end;
      
      function setdetectormode(det)
         mode = EPS1000.getcontrolmode;
         mode.detector = det;
         EPS1000.setcontrolmode(mode);
      end;
      
      function detmode = getdetectormode()
         mode = EPS1000.getcontrolmode;
         detmode = mode.detector;
      end;
      
      function setgradtype(type)
         mode = EPS1000.getcontrolmode;
         mode.gradtype = type;
         EPS1000.setcontrolmode(mode);
      end;
      
      function gradtype = getgradtype()
         mode = EPS1000.getcontrolmode;
         gradtype = mode.gradtype;
      end;
      
      
      
      
      function [fb, fb_dBm] = getfeedback(detector) % detector 0 or 1

         if detector==0,
         
            fb = EPS1000.read(350);
            bs  = mod(EPS1000.read(352), 2^8); %possible bit-shift, is 0xFF if not supported
            if bs<255,
               fb = fb / 2^bs;
            end;
            fb_dBm = 0;
            
            Feedback_IntMax_uW = EPS1000.read(353);
            if (Feedback_IntMax_uW>0) && (Feedback_IntMax_uW<65535)
               Feedback_Idark0 = EPS1000.read(355);
               Pow_uW = fb / (2 ^ 16 - Feedback_Idark0) * Feedback_IntMax_uW;
               fb_dBm = 10 * log10(Pow_uW / 1000);
            end;
            
         else

            fb = EPS1000.read(351);
            bs  = bitshift(EPS1000.read(352), -8); %possible bit-shift, is 0xFF if not supported
            if bs<255,
               fb = fb / 2^bs;
            end;
         
            Referencee_IntMax_uW = EPS1000.read(354);
            if (Referencee_IntMax_uW>0) && (Referencee_IntMax_uW<65535)
               Reference_Idark0 = EPS1000.read(359);
               Pow_uW = fb / (2 ^ 16 - Reference_Idark0) * Referencee_IntMax_uW;
               fb_dBm = 10 * log10(Pow_uW / 1000);
            end;
            
         end;

      end;
      
      % forces the receiver dut path into a sensitive mode
      function setsensmodedut(mode)
         AE32_state = bitand( EPS1000.read(126), bin2dec('1100')); % get BIT2 and BIT3
         manual = 2^7 + 2^5 + 2^1; % BIT7, BIT5 and BIT1 for manual mode of corresponding power range
         switch mode
            case 0 % normal mode, -2 dBm
               SRA4 = 0; SRA3 = 0; AC22 = 0;
            case 1 %sens. mode 1, -17 dBm
               SRA4 = 0; SRA3 = 1; AC22 = 1;
            case 2 % sens. mode 2, -32 dBm
               SRA4 = 1; SRA3 = 0; AC22 = 1;
            case 3 % sens. mode 3, -47 dBm
               SRA4 = 0; SRA3 = 0; AC22 = 1;
            otherwise % set to automatic
               SRA4 = 0; SRA3 = 0; AC22 = 0; manual = 0;
         end
         EPS1000.write(126, manual + SRA4*2^6 + SRA3*2^4 + AC22*2^0 + AE32_state);
      end;
      
      % forces the receiver ref path into a sensitive mode
      function setsensmoderef(mode)
         reg = EPS1000.read(126);
         SRA4_state = bitand( reg, bin2dec('11000000'));
         SRA3_state = bitand( reg, bin2dec('110000'));
         AC22_state = bitand( reg, bin2dec('11'));
         manual = 2^3;
         switch mode
            case 0 % normal mode, -2 dBm
               AE32 = 0;
            case 1 %sens. mode 1, -17 dBm
               AE32 = 1;
            otherwise % set to automatic
               AE32 = 1;
               manual = 0;
         end;
         EPS1000.write(126, manual + SRA4_state + SRA3_state + AE32*2^2 + AC22_state);
      end;
      
      % calibrate the dark current of the ADCs
      function setdarkcurrent
      
         
         mode = EPS1000.getcontrolmode;
         %mode_save = mode;

         if mode.detector>=1
            input("Turn off optical input signal and press Enter...");
         else
            disp("Nothing to calibrate in detector mode 0."); 
            return;
         end;
         
         %mode.sensitive = 0;
         %EPS1000.setcontrolmode(mode);
         
         % calibrate only the detector mode that is active
         switch mode.detector
         
            case 1 % "optical"
         
               for ii=1:4
                  EPS1000.write(355-1+ii, 0);
                  EPS1000.setsensmodedut(ii-1);
                  pause(0.1);
                  Idark_adc1_dut(ii) = round(mean(EPS1000.readburst(380, 1, 2^12, 380)));
               end;
               for ii=1:4
                  EPS1000.write(355-1+ii, Idark_adc1_dut(ii)-2);
               end;
         
               for ii=1:2
                  EPS1000.write(359-1+ii, 0);
                  EPS1000.setsensmoderef(ii-1);
                  pause(0.1);
                  Idark_adc2_ref(ii) = round(mean(EPS1000.readburst(381, 1, 2^12, 381)));
               end;
               for ii=1:2
                  EPS1000.write(359-1+ii, Idark_adc2_ref(ii));
               end;
         
               Idark_adc1_dut
               Idark_adc2_ref
               
            case 2 % "electrical logarithmic 0..5V
               
               EPS1000.write(369, 0);
               Idark_adc2_electrical1 = round(mean(EPS1000.readburst(381, 1, 2^12, 381)));
               EPS1000.write(369, Idark_adc2_electrical1);
               
               Idark_adc2_electrical1
               
            case 3 % "electrical 0..2V"
            
               EPS1000.write(361, 0);
               EPS1000.setsensmoderef(0);
               pause(0.1);
               Idark_adc1_ref(1) = round(mean(EPS1000.readburst(380, 1, 2^12, 380)));
               EPS1000.write(361, Idark_adc1_ref(1));
         
               EPS1000.write(368, 0);
               EPS1000.setsensmoderef(1);
               pause(0.1);
               Idark_adc1_ref(2) = round(mean(EPS1000.readburst(380, 1, 2^12, 380)));
               EPS1000.write(368, Idark_adc1_ref(2));
         
               EPS1000.setsensmodedut(-1); % must be set to auto for Idark_adc2_electrical2
         
               EPS1000.write(378, 0);
               Idark_adc2_electrical2 = round(mean(EPS1000.readburst(381, 1, 2^12, 381)));
               figure; plot(EPS1000.readburst(381, 1, 2^12, 381));
               EPS1000.write(378, Idark_adc2_electrical2);
         
               Idark_adc1_ref
               Idark_adc2_electrical2
               
            otherwise
         end;
         
         % clean up
         EPS1000.setsensmodedut(-1); % set to auto
         EPS1000.setsensmoderef(-1); % set to auto
         
         %EPS1000.setcontrolmode(mode_save); % restore initial configuration
         
         EPS1000.write(290, 209 * 2^8 + 1); % store in Flash
         
      end;
      
      
      
      % histogram during control
      function sethistogramoffset(offset)
         EPS1000.write(337, offset);
      end;
      
      function histogramstart()
         EPS1000.write(337, 1);
      end;
      
      function histogramstop()
         EPS1000.write(337, 2);
      end;
      
      function hist = histogramload()
         hist_help1 = EPS1000.readburst(339, 2^15+0, 2^15+1023, 339);
         hist_help2 = EPS1000.readburst(339, 2^14+0, 2^14+1023, 339);
         hist_help3 = EPS1000.readburst(339,      0,      1023, 339);
         for i=1:1024,
            hist(i) = hist_help1(i)*2^32 + hist_help2(i)*2^16 + hist_help3(i);
         end;
      end;
      
      
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      % PDL measurement
      % requires tracking option with optical feedback and fw>=1227
      %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      
      function r = get_pdl_syncstate
         r = bitand(EPS1000.read(334), 2^3);
      end;
      
      function set_pdl_ext_enable(en, ref, pdltype)

         if nargin<2,
            ref = 0;
         end;
      
         if nargin<3,
            pdltype = "HIGH";
         end;
         
         if en==1,
            mode = EPS1000.getcontrolmode();
            
            if pdltype=="HIGH",
               mode.sensitive  = 3;
               %mode.sensitive  = 0;
               DITH = 4;
               %DITH = 6;
               ATE = 4;              
               %ATE = 3;              
            elseif pdltype=="LOW",
               DITH = 7;
               ATE = 5;
            else
               disp("Argument 3 (pdltype) must either be ""LOW"" or ""HIGH"""); 
               return;
            end;
            
            detmode = EPS1000.getdetectormode;
            if detmode==3,
               ATE = 11; % detector mode 3
            end;
            
            EPS1000.write(344, 31); % delay
            EPS1000.write(343, ATE); % averaging time exponent
            EPS1000.write(345, DITH); % dither amplitude
         
         
            EPS1000.write(340, 6); % set tracking mode and enable pdl_ext

            mode.gradtype   = 1;
            mode.reference  = ref;

            EPS1000.setcontrolmode(mode);
      
         else
            dummy = EPS1000.read(340);
            EPS1000.write(340, bitand(dummy,0xFB)); % disable pdl_ext but leave EPS in tracking mode
         end;
      end;
      
      function set_pdl_ext_conf(meas_exp, niter)
         % default:           niter = 150; meas_exp = 15
         % fast measurement:  niter = 100; meas_exp = 12
         EPS1000.write(335, meas_exp*2^11 + niter);
      end;
      
      function [PDL_dB, ML_dB, HL_dB, LL_dB, minint_debug] = getpdl_ext(withref)
         maxint    = EPS1000.read(370); % this latches registers 371 and 372 and locks registers 370 to 372
         maxintref = EPS1000.read(371);
         bsreg  = EPS1000.read(372);    % this releases registers 370 to 372
         bs     = mod(bsreg, 2^8);
         bsref  = bitshift(bsreg, -8);
         maxint    = maxint / 2^bs;
         maxintref = maxintref / 2^bsref;
 
         minint    = EPS1000.read(373);  % this latches registers 374 and 375 and locks registers 373 to 375
         minint_debug = minint;
         minintref = EPS1000.read(374);
         bsreg  = EPS1000.read(375);     % this releases registers 373 to 375
         bs     = mod(bsreg, 2^8);
         bsref  = bitshift(bsreg, -8);
         minint    = minint / 2^bs;
         minintref = minintref / 2^bsref;
      
         if withref==0
            LL  = 1;
            HL  = minint/maxint;
         else
            HL = minint/minintref;
            LL = maxint/maxintref;
         end

         ML = (HL+LL)/2;
         ML_dB  = -10*log10(ML);
         HL_dB  = -10*log10(HL);
         LL_dB  = -10*log10(LL);
         PDL_dB = HL_dB-LL_dB;

      end
      
      function set_pdl_dep_enable(en) % this disables tracking mode!
         if en==1,
            EPS1000.write(340, 8);  % enable pdl_dep measurement
            EPS1000.write(150, 15); % define rotation speeds by "rotations per time"
            EPS1000.setrotation_all(ones(7,1)); % enable rotation for all WPs
            
            % set reqired control mode
            mode = EPS1000.getcontrolmode();
            mode.detector  = 1;
            mode.sensitive = 0;
            EPS1000.setcontrolmode(mode);
            
         else
            EPS1000.write(340, 0); % disable pdl_dep measurement
            EPS1000.write(150, 14); % define rotation speeds by regular speed registers
            EPS1000.setrotation_all(zeros(7,1)); % disable rotation for all WPs
         end;
            
      end;
      
      
      function [PDL_dB, ML_dB] = getpdl_dep_alt(withref)
         PDL_dB = EPS1000.read(362)/1024;
         if withref==0,
            ML_dB = 0;
         else
            meandut = EPS1000.read(363);
            meanref = EPS1000.read(365);
            ML_dB = 10*log10(meanref/meandut);
         end;
      end;
      
      function [PDL_dB, ML_dB, HL_dB, LL_dB] = getpdl_dep(withref)
         PDL_dB = EPS1000.read(362)*9.765625e-04;
         if withref==0
            LL_dB = 0;            LL = 1;
            HL_dB = PDL_dB;       HL = 10^(-0.1*HL_dB);
            ML    = 0.5*(HL+LL);  ML_dB = -10*log10(ML);
         else
            meandut = EPS1000.read(363);
            meanref = EPS1000.read(365);
            ML    = meandut/meanref;
            LL    = ML*2/(10^(-0.1*PDL_dB)+1);
            LL_dB = -10*log10(LL);
            HL_dB = LL_dB+PDL_dB;
            ML_dB = -10*log10(ML);
         end
      end
      
      function [dut, ref] = getpdlmemory(naddr, withref)
      
         dut   = EPS1000.readburst(330, 0, naddr-1, 330);
         bsreg = EPS1000.readburst(330, 0, naddr-1, 332);
         bs     = mod(bsreg, 2^8);
         bsref  = bitshift(bsreg, -8);
         dut    = dut ./ 2.^bs;
         
         if withref,
            ref   = EPS1000.readburst(330, 0, naddr-1, 331);
            ref = ref ./ 2.^bsref;
         else
            ref = ones(naddr);
         end;
         
      end;
      

   end;

end